<?php

namespace App\Http\Controllers\apps;

use App\Http\Controllers\Controller;
use App\Models\Pagamento;
use App\Models\Cliente;
use Illuminate\Http\Request;
use App\Models\PlanoRenovacao;
use Illuminate\Support\Facades\Auth;
use App\Models\CompanyDetail;
use App\Http\Controllers\SendMessageController;
use App\Models\Plano;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use App\Models\Template;
use App\Models\User;
use Carbon\Carbon;

class EcommerceOrderDetails extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    public function index(Request $request)
    {
        $user = Auth::user();
        $paymentId = $request->query('order_id');
        $payment = Pagamento::find($paymentId);

        if (!$payment) {
            return redirect()->back()->with('error', 'Pagamento não encontrado.');
        }

        $cliente = Cliente::find($payment->cliente_id);
        if (!$cliente) {
            return redirect()->back()->with('error', 'Nenhum Pagamento encontrado para este Cliente.');
        }

        $empresa = CompanyDetail::where('user_id', $payment->user_id)->first();
        if (!$empresa) {
            return redirect()->back()->with('error', 'Empresa não encontrada.');
        }

        $plano = Plano::find($cliente->plano_id);
        if (!$plano) {
            return redirect()->back()->with('error', 'Plano não encontrado.');
        }

        $planos_revenda = PlanoRenovacao::all();
        $current_plan_id = $user->plano_id;

        return view('content.apps.detalhes', compact(
            'payment',
            'cliente',
            'planos_revenda',
            'current_plan_id',
            'empresa',
            'plano'
        ));
    }

    // ✅ FUNÇÃO SUBSTITUÍDA
    public function addPayment(Request $request)
    {
        DB::transaction(function () use ($request) {
            $payment = Pagamento::with('cliente.plano')->findOrFail($request->payment_id);

            $payment->update([
                'valor' => $request->invoiceAmount,
                'status' => $request->payment_status,
                'payment_date' => $request->payment_date
            ]);

            if ($payment->status === 'approved' && $payment->cliente) {
                $this->processApprovedPayment($payment);
            }
        });

        return redirect()->back()->with('success', 'Pagamento atualizado com sucesso.');
    }

    // ✅ FUNÇÃO SUBSTITUÍDA
    protected function processApprovedPayment(Pagamento $payment)
    {
        $cliente = $payment->cliente;
        $plano = $cliente->plano;
        $vencimento = Carbon::parse($cliente->vencimento);
        $paymentDate = Carbon::parse($payment->payment_date);

        if ($vencimento->lt($paymentDate)) {
            $vencimento = $paymentDate;
        }

        $diasParaAdicionar = $this->calcularDiasRenovacao($plano, $paymentDate);
        $novoVencimento = $vencimento->addDays($diasParaAdicionar);

        $cliente->update([
            'vencimento' => $novoVencimento
        ]);

        try {
            $this->notifyClientAndOwner($payment, true);
        } catch (\Exception $e) {
            \Log::error('Falha nas notificações', [
                'error' => $e->getMessage(),
                'payment_id' => $payment->id
            ]);
        }
    }

    private function notifyClientAndOwner($paymentRecord, $shouldProcessRenewal = false)
    {
        $cliente = Cliente::find($paymentRecord->cliente_id);
        if (!$cliente) return;

        $template = Template::where('finalidade', 'pagamentos')
            ->where('user_id', $cliente->user_id)
            ->first();

        if (!$template) {
            $template = Template::where('finalidade', 'pagamentos')
                ->whereNull('user_id')
                ->firstOrFail();
        }

        $statusPagamentoMap = [
            'paid' => 'Pago',
            'pending' => 'Pendente',
            'failed' => 'Falhou',
            'in_process' => 'Em Processo',
            'approved' => 'Aprovado',
        ];

        $statusPagamento = $statusPagamentoMap[$paymentRecord->status] ?? $paymentRecord->status ?? 'Status do Pagamento';

        $company = CompanyDetail::where('user_id', $cliente->user_id)->first();
        $nomeEmpresa = $company ? $company->company_name : '{nome_empresa}';
        $whatsappEmpresa = $company ? $company->company_whatsapp : '{whatsapp_empresa}';

        $owner = User::find($cliente->user_id);
        $nomeDono = $owner ? $owner->name : '{nome_dono}';
        $whatsappDono = $owner ? $owner->whatsapp : '{whatsapp_dono}';

        $plano = Plano::find($paymentRecord->plano_id);
        $nomePlano = $plano ? $plano->nome : 'Nome do Plano';
        $valorPlano = $plano ? $plano->preco : 'Valor do Plano';

        $saudacao = $this->getSaudacao();
        $textExpirate = $this->getTextExpirate($cliente->vencimento);

        $dadosCliente = [
            '{nome_cliente}' => $cliente->nome ?? 'Nome do Cliente',
            '{telefone_cliente}' => $cliente->whatsapp ?? '(11) 99999-9999',
            '{notas}' => $cliente->notas ?? 'Notas',
            '{vencimento_cliente}' => Carbon::parse($cliente->vencimento)->format('d/m/Y') ?? 'Vencimento do Cliente',
            '{plano_nome}' => $nomePlano,
            '{plano_valor}' => $valorPlano,
            '{data_atual}' => Carbon::now()->format('d/m/Y'),
            '{plano_link}' => $paymentRecord->link_pagamento ?? 'Link de Pagamento',
            '{text_expirate}' => $textExpirate,
            '{saudacao}' => $saudacao,
            '{payload_pix}' => $paymentRecord->payload_pix ?? 'Pix Copia e Cola',
            '{whatsap_empresa}' => $whatsappEmpresa,
            '{status_pagamento}' => $statusPagamento,
            '{nome_empresa}' => $nomeEmpresa,
            '{nome_dono}' => $nomeDono,
        ];

        $conteudoCliente = $this->substituirPlaceholders($template->conteudo, $dadosCliente);
        $this->sendMessage($cliente->whatsapp, $conteudoCliente, $cliente->user_id);

        if ($owner) {
            $creditos = 1;
            $renovacaoSucesso = true;
            $mensagemErro = '';

            if ($shouldProcessRenewal && $cliente->sync_qpanel == 1 && $paymentRecord->status === 'approved') {
                $infoPlano = $this->obterCreditosPlanoQPanel($cliente->plano_qpanel);
                $creditos = $infoPlano['credits'];
                $resultadoRenovacao = $this->renovarNoQPanel($cliente);
                $renovacaoSucesso = $resultadoRenovacao['success'];
                $mensagemErro = $resultadoRenovacao['message'] ?? '';
            }

            $mensagemDono = "Olá, tudo bem?\n";
            $mensagemDono .= "O cliente {$cliente->nome} fez o pagamento do plano *{$nomePlano}*.\n";
            $mensagemDono .= "No valor de: R$ {$valorPlano}.\n";
            $mensagemDono .= "Data do Pagamento: " . Carbon::now()->format('d/m/Y') . ".\n";
            $mensagemDono .= "Nova data de vencimento: " . Carbon::parse($cliente->vencimento)->format('d/m/Y') . ".\n\n";

            if ($cliente->sync_qpanel == 1) {
                if ($renovacaoSucesso) {
                    $mensagemDono .= "Seu cliente foi renovado no Qpanel e foi deduzido {$creditos} crédito" . ($creditos > 1 ? 's' : '') . " do seu painel.";
                } else {
                    if (str_contains($mensagemErro, 'You don\'t have enough credits')) {
                        $mensagemDono .= "Entretanto, o seu painel não possui créditos suficientes para fazer a renovação automática.";
                    } else {
                        $mensagemDono .= "Houve um erro ao renovar no QPanel: {$mensagemErro}";
                    }
                }
            }

            $this->sendMessage($owner->whatsapp, $mensagemDono, $owner->id);
        }
    }

    private function renovarNoQPanel($cliente)
    {
        if ($cliente->sync_qpanel != 1 || empty($cliente->iptv_nome) || empty($cliente->plano_qpanel)) {
            return ['success' => false, 'message' => 'Cliente não configurado para sincronização com QPanel'];
        }

        $dono = User::find($cliente->user_id);
        if (!$dono || empty($dono->id_qpanel)) {
            return ['success' => false, 'message' => 'Dono do cliente não configurado no QPanel'];
        }

        try {
            $companyDetails = CompanyDetail::where('user_id', 1)->first();

            if (!$companyDetails || !$companyDetails->qpanel_api_url || !$companyDetails->qpanel_api_key) {
                Log::error('Credenciais do QPanel não configuradas no sistema');
                return ['success' => false, 'message' => 'Configurações do QPanel não encontradas'];
            }

            $curl = curl_init();
            $postData = [
                'userId' => $dono->id_qpanel,
                'username' => $cliente->iptv_nome,
                'packageId' => $cliente->plano_qpanel
            ];
            $urlCompleta = rtrim($companyDetails->qpanel_api_url, '/') . '/api/webhook/customer/renew';

            curl_setopt_array($curl, [
                CURLOPT_URL => $urlCompleta,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_ENCODING => '',
                CURLOPT_MAXREDIRS => 10,
                CURLOPT_TIMEOUT => 30,
                CURLOPT_FOLLOWLOCATION => true,
                CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
                CURLOPT_CUSTOMREQUEST => 'POST',
                CURLOPT_POSTFIELDS => json_encode($postData),
                CURLOPT_HTTPHEADER => [
                    'Content-Type: application/json',
                    'Authorization: Bearer ' . $companyDetails->qpanel_api_key
                ],
            ]);

            $response = curl_exec($curl);
            $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
            curl_close($curl);

            $responseData = json_decode($response, true);

            if ($httpCode !== 200) {
                Log::error('Falha ao renovar no QPanel', [
                    'cliente_id' => $cliente->id,
                    'response' => $response,
                    'http_code' => $httpCode,
                    'api_url' => $urlCompleta
                ]);
                return ['success' => false, 'message' => $responseData['message'] ?? 'Erro ao renovar no QPanel'];
            }

            return ['success' => true, 'message' => 'Renovação no QPanel realizada com sucesso'];

        } catch (\Exception $e) {
            Log::error('Exceção ao renovar no QPanel', [
                'cliente_id' => $cliente->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }

    // ✅ FUNÇÃO SUBSTITUÍDA
    private function calcularDiasRenovacao($plano, $paymentDate)
    {
        switch ($plano->tipo_duracao) {
            case 'meses':
                return $paymentDate->daysInMonth;

            case 'anos':
                return 365;

            case 'dias':
            default:
                return $plano->duracao_em_dias;
        }
    }

    private function getTextExpirate($vencimento)
    {
        $dataVencimento = Carbon::parse($vencimento);
        $dataAtual = Carbon::now();
        $intervalo = $dataAtual->diff($dataVencimento);

        if ($intervalo->invert) {
            return 'expirou há ' . $intervalo->days . ' dias';
        } elseif ($intervalo->days == 0) {
            return 'expira hoje';
        } else {
            return 'expira em ' . $intervalo->days . ' dias';
        }
    }

    private function getSaudacao()
    {
        $hora = date('H');
        if ($hora < 12) {
            return 'Bom dia!';
        } elseif ($hora < 18) {
            return 'Boa tarde!';
        } else {
            return 'Boa noite!';
        }
    }

    private function substituirPlaceholders($conteudo, $dados)
    {
        foreach ($dados as $placeholder => $valor) {
            $conteudo = str_replace($placeholder, $valor, $conteudo);
        }
        return $conteudo;
    }

    private function sendMessage($phone, $message, $user_id)
    {
        $sendMessageController = new SendMessageController();
        $request = new Request([
            'phone' => $phone,
            'message' => $message,
            'user_id' => $user_id,
        ]);
        $sendMessageController->sendMessageWithoutAuth($request);
    }
}
